const fs = require('fs');
const path = require('path');
const config = require('./config');

// Log file for session history
const LOG_FILE = path.join(__dirname, 'session.log');

const log = (msg) => {
  const timestamp = new Date().toLocaleTimeString();
  const logLine = `[${timestamp}] ${msg}`;
  console.log(logLine);
  
  // Also write to log file
  try {
    fs.appendFileSync(LOG_FILE, logLine + '\n');
  } catch (e) {
    // Ignore log file errors
  }
};

const sleep = (ms) => new Promise(resolve => setTimeout(resolve, ms));

const randomDelay = (min, max) => {
  const delay = Math.floor(Math.random() * (max - min + 1)) + min;
  return sleep(delay);
};

const loadUsers = () => {
  if (!fs.existsSync(config.USERS_FILE)) {
    return [];
  }
  const content = fs.readFileSync(config.USERS_FILE, 'utf-8');
  return content.split('\n').map(u => u.trim()).filter(u => u.length > 0);
};

const saveUsers = (users) => {
  fs.writeFileSync(config.USERS_FILE, users.join('\n'));
};

const markProcessed = (username) => {
  const timestamp = new Date().toISOString();
  fs.appendFileSync(config.PROCESSED_FILE, `${username}\t${timestamp}\n`);
};

// Get count of remaining users
const getUserCount = () => {
  if (!fs.existsSync(config.USERS_FILE)) {
    return 0;
  }
  const content = fs.readFileSync(config.USERS_FILE, 'utf-8');
  return content.split('\n').filter(u => u.trim().length > 0).length;
};

// Get count of processed users
const getProcessedCount = () => {
  if (!fs.existsSync(config.PROCESSED_FILE)) {
    return 0;
  }
  const content = fs.readFileSync(config.PROCESSED_FILE, 'utf-8');
  return content.split('\n').filter(u => u.trim().length > 0).length;
};

// Clear session log at start of new session
const clearSessionLog = () => {
  const header = `=== UnjankyIG Session Started: ${new Date().toLocaleString()} ===\n`;
  fs.writeFileSync(LOG_FILE, header);
};

module.exports = {
  log,
  sleep,
  randomDelay,
  loadUsers,
  saveUsers,
  markProcessed,
  getUserCount,
  getProcessedCount,
  clearSessionLog
};
