/**
 * UnjankyIG - Stripe Payment Service
 * Custom payment integration for license activation
 */

require('dotenv').config();
const crypto = require('crypto');

// Initialize Stripe
let stripe;
try {
  if (!process.env.STRIPE_SECRET_KEY) {
    console.warn('⚠️  STRIPE_SECRET_KEY not found in .env file');
  } else {
    stripe = require('stripe')(process.env.STRIPE_SECRET_KEY);
  }
} catch (error) {
  console.warn('⚠️  Stripe module not initialized:', error.message);
}

// Product Configuration
const PRODUCTS = {
  WINDOWS: {
    name: 'UnjankyIG - Windows License',
    price: 799, // $7.99 in cents
    currency: 'usd',
    priceId: process.env.WINDOWS_PRICE_ID,
  },
  MAC: {
    name: 'UnjankyIG - macOS License',
    price: 999, // $9.99 in cents
    currency: 'usd',
    priceId: process.env.MAC_PRICE_ID,
  }
};

/**
 * Generate a unique license key
 */
function generateLicenseKey() {
  const timestamp = Date.now().toString(36).toUpperCase();
  const random = crypto.randomBytes(8).toString('hex').toUpperCase();
  const checksum = crypto
    .createHash('md5')
    .update(timestamp + random)
    .digest('hex')
    .substring(0, 4)
    .toUpperCase();
  
  return `UJIG-${timestamp}-${random.substring(0, 8)}-${checksum}`;
}

/**
 * Validate a license key format
 */
function validateLicenseKeyFormat(key) {
  const pattern = /^UJIG-[A-Z0-9]+-[A-Z0-9]{8}-[A-Z0-9]{4}$/;
  return pattern.test(key);
}

/**
 * Create a payment session for Stripe Checkout
 */
async function createPaymentSession(platform = 'WINDOWS', customerEmail = null) {
  if (!stripe) {
    throw new Error('Stripe is not initialized. Check your API keys.');
  }

  const product = PRODUCTS[platform] || PRODUCTS.WINDOWS;
  const licenseKey = generateLicenseKey();

  // Create checkout session
  const session = await stripe.checkout.sessions.create({
    payment_method_types: ['card'],
    line_items: [
      {
        price_data: {
          currency: product.currency,
          product_data: {
            name: product.name,
            description: 'Unlimited unfollows after 200 free trial',
            metadata: {
              platform: platform,
              licenseKey: licenseKey,
            }
          },
          unit_amount: product.price,
        },
        quantity: 1,
      },
    ],
    mode: 'payment',
    success_url: `${process.env.DASHBOARD_URL || 'http://localhost:3456'}/payment-success?session_id={CHECKOUT_SESSION_ID}&key=${licenseKey}`,
    cancel_url: `${process.env.DASHBOARD_URL || 'http://localhost:3456'}/payment-cancelled`,
    customer_email: customerEmail,
    metadata: {
      licenseKey: licenseKey,
      platform: platform,
      product: 'UnjankyIG',
    },
  });

  return {
    sessionId: session.id,
    url: session.url,
    licenseKey: licenseKey,
  };
}

/**
 * Verify a payment session was completed
 */
async function verifyPaymentSession(sessionId) {
  if (!stripe) {
    throw new Error('Stripe is not initialized. Check your API keys.');
  }

  const session = await stripe.checkout.sessions.retrieve(sessionId);
  
  if (session.payment_status === 'paid') {
    return {
      success: true,
      licenseKey: session.metadata.licenseKey,
      platform: session.metadata.platform,
      customerEmail: session.customer_email,
      amount: session.amount_total,
    };
  }

  return {
    success: false,
    status: session.payment_status,
  };
}

/**
 * Get payment details by license key (searches for it)
 */
async function getPaymentByLicenseKey(licenseKey) {
  if (!stripe) {
    throw new Error('Stripe is not initialized. Check your API keys.');
  }

  const sessions = await stripe.checkout.sessions.list({
    limit: 100,
  });

  const matchingSession = sessions.data.find(
    s => s.metadata.licenseKey === licenseKey && s.payment_status === 'paid'
  );

  if (matchingSession) {
    return {
      found: true,
      sessionId: matchingSession.id,
      customerEmail: matchingSession.customer_email,
      amount: matchingSession.amount_total,
      platform: matchingSession.metadata.platform,
    };
  }

  return { found: false };
}

/**
 * Handle Stripe webhook events
 */
async function handleWebhook(payload, signature) {
  if (!stripe || !process.env.STRIPE_WEBHOOK_SECRET) {
    throw new Error('Stripe webhook not configured');
  }

  const event = stripe.webhooks.constructEvent(
    payload,
    signature,
    process.env.STRIPE_WEBHOOK_SECRET
  );

  switch (event.type) {
    case 'checkout.session.completed':
      const session = event.data.object;
      console.log('✅ Payment completed:', session.metadata.licenseKey);
      // You can add logic here to automatically activate licenses
      return { success: true, licenseKey: session.metadata.licenseKey };

    case 'payment_intent.payment_failed':
      console.log('❌ Payment failed:', event.data.object.id);
      return { success: false, reason: 'payment_failed' };

    default:
      console.log('ℹ️  Unhandled event type:', event.type);
  }

  return { success: true };
}

/**
 * Create or update products and prices in Stripe
 */
async function setupStripeProducts() {
  if (!stripe) {
    throw new Error('Stripe is not initialized. Check your API keys.');
  }

  console.log('🔧 Setting up Stripe products...');

  // Create Windows product
  const windowsProduct = await stripe.products.create({
    name: PRODUCTS.WINDOWS.name,
    description: 'UnjankyIG for Windows - Unlimited Instagram unfollows after 200 free trial',
    metadata: {
      platform: 'WINDOWS',
      freeLimit: '200',
    }
  });

  const windowsPrice = await stripe.prices.create({
    product: windowsProduct.id,
    unit_amount: PRODUCTS.WINDOWS.price,
    currency: PRODUCTS.WINDOWS.currency,
    metadata: {
      platform: 'WINDOWS',
    }
  });

  // Create Mac product
  const macProduct = await stripe.products.create({
    name: PRODUCTS.MAC.name,
    description: 'UnjankyIG for macOS - Unlimited Instagram unfollows after 200 free trial',
    metadata: {
      platform: 'MAC',
      freeLimit: '200',
    }
  });

  const macPrice = await stripe.prices.create({
    product: macProduct.id,
    unit_amount: PRODUCTS.MAC.price,
    currency: PRODUCTS.MAC.currency,
    metadata: {
      platform: 'MAC',
    }
  });

  console.log('✅ Products created successfully!');
  console.log('\n📋 Add these to your .env file:');
  console.log(`WINDOWS_PRICE_ID=${windowsPrice.id}`);
  console.log(`MAC_PRICE_ID=${macPrice.id}`);

  return {
    windows: { productId: windowsProduct.id, priceId: windowsPrice.id },
    mac: { productId: macProduct.id, priceId: macPrice.id },
  };
}

module.exports = {
  generateLicenseKey,
  validateLicenseKeyFormat,
  createPaymentSession,
  verifyPaymentSession,
  getPaymentByLicenseKey,
  handleWebhook,
  setupStripeProducts,
  PRODUCTS,
};

