/**
 * License Manager for CLI Fallback Version
 * Syncs with the Premium app's license system
 */

const fs = require('fs');
const path = require('path');
const os = require('os');

const LICENSE_DIR = path.join(os.homedir(), '.unjankyourinstagram');
const LICENSE_FILE = path.join(LICENSE_DIR, 'license.json');
const UNFOLLOW_COUNTER_FILE = path.join(LICENSE_DIR, 'unfollow_counter.json');
const FREE_TRIAL_LIMIT = 200; // Reduced from 500 to 200

class LicenseManager {
  constructor() {
    this.ensureDir();
  }

  ensureDir() {
    if (!fs.existsSync(LICENSE_DIR)) {
      fs.mkdirSync(LICENSE_DIR, { recursive: true });
    }
  }

  getLicense() {
    try {
      if (fs.existsSync(LICENSE_FILE)) {
        const data = fs.readFileSync(LICENSE_FILE, 'utf8');
        return JSON.parse(data);
      }
    } catch (e) {
      // Silent fail
    }
    return { isPremium: false };
  }

  getUnfollowCounter() {
    try {
      if (fs.existsSync(UNFOLLOW_COUNTER_FILE)) {
        const data = fs.readFileSync(UNFOLLOW_COUNTER_FILE, 'utf8');
        const counter = JSON.parse(data);
        
        // Verify checksum
        if (this.verifyChecksum(counter)) {
          return counter;
        } else {
          // Tampering detected - max it out
          console.log('🚨 Tampering detected! Counter reset to maximum.');
          const punishment = {
            lifetimeCount: FREE_TRIAL_LIMIT,
            lastUpdated: new Date().toISOString(),
            checksum: ''
          };
          punishment.checksum = this.generateChecksum(punishment.lifetimeCount, punishment.lastUpdated);
          this.saveUnfollowCounter(punishment);
          return punishment;
        }
      }
    } catch (e) {
      // Silent fail
    }
    
    const newCounter = {
      lifetimeCount: 0,
      lastUpdated: new Date().toISOString(),
      checksum: ''
    };
    newCounter.checksum = this.generateChecksum(newCounter.lifetimeCount, newCounter.lastUpdated);
    return newCounter;
  }

  saveUnfollowCounter(counter) {
    try {
      this.ensureDir();
      fs.writeFileSync(UNFOLLOW_COUNTER_FILE, JSON.stringify(counter, null, 2));
    } catch (e) {
      console.error('Warning: Could not save unfollow counter');
    }
  }

  generateChecksum(count, timestamp) {
    const secret = 'JMFG_UNJANKY_2025_SALT_XK9P';
    const data = `${count}:${timestamp}:${secret}`;
    let hash = 0;
    for (let i = 0; i < data.length; i++) {
      const char = data.charCodeAt(i);
      hash = ((hash << 5) - hash) + char;
      hash = hash & hash;
    }
    return Math.abs(hash).toString(36);
  }

  verifyChecksum(counter) {
    const expectedChecksum = this.generateChecksum(counter.lifetimeCount, counter.lastUpdated);
    return counter.checksum === expectedChecksum;
  }

  getRemainingUnfollows() {
    const license = this.getLicense();
    if (license.isPremium) return Infinity;
    
    const counter = this.getUnfollowCounter();
    return Math.max(0, FREE_TRIAL_LIMIT - counter.lifetimeCount);
  }

  canUnfollowMore() {
    const license = this.getLicense();
    if (license.isPremium) return true;
    
    const counter = this.getUnfollowCounter();
    return counter.lifetimeCount < FREE_TRIAL_LIMIT;
  }

  incrementUnfollowCount() {
    const counter = this.getUnfollowCounter();
    counter.lifetimeCount += 1;
    counter.lastUpdated = new Date().toISOString();
    counter.checksum = this.generateChecksum(counter.lifetimeCount, counter.lastUpdated);
    this.saveUnfollowCounter(counter);
    return counter.lifetimeCount;
  }

  isPremium() {
    return this.getLicense().isPremium;
  }
}

module.exports = new LicenseManager();

