const puppeteer = require('puppeteer');
const { exec } = require('child_process');
const fs = require('fs');
const path = require('path');
const config = require('./config');
const utils = require('./utils');
const { generateDailyReport } = require('./reporter');
const licenseManager = require('./license_manager');
const CrashProofPurge = require('./crash_proof_purge');

// Graceful shutdown handling
let shouldStop = false;
let browser = null;
let page = null;

// Initialize crash-proof purge system
const purgeSystem = new CrashProofPurge();

// Cleanup function to close browser
async function cleanup() {
  if (browser) {
    try {
      await browser.close();
    } catch (e) {
      // Force kill Chrome if graceful close fails
      try {
        const { exec } = require('child_process');
        exec('taskkill /F /IM chrome.exe /T', () => {});
      } catch (e2) {}
    }
    browser = null;
  }
}

process.on('SIGINT', async () => {
  utils.log('\n⚠️ Shutdown requested... finishing current user...');
  shouldStop = true;
});

process.on('SIGTERM', async () => {
  utils.log('\n⚠️ Shutdown requested... finishing current user...');
  shouldStop = true;
});

// Make sure Chrome closes if program crashes or exits unexpectedly
process.on('exit', () => {
  if (browser) {
    try { browser.close(); } catch(e) {}
  }
});

process.on('uncaughtException', async (err) => {
  utils.log(`❌ Unexpected error: ${err.message}`);
  await cleanup();
  process.exit(1);
});

process.on('unhandledRejection', async (err) => {
  utils.log(`❌ Unexpected error: ${err}`);
  await cleanup();
  process.exit(1);
});

// ============================================
// CHROME MANAGEMENT SYSTEM
// ============================================

const BROWSER_ARGS = [
  // Stability first
  '--no-first-run',
  '--no-default-browser-check',
  '--disable-popup-blocking',
  '--disable-prompt-on-repost',
  '--disable-sync',
  '--disable-features=TranslateUI',
  
  // Performance (less aggressive)
  '--disable-background-timer-throttling',
  '--disable-backgrounding-occluded-windows',
  '--disable-renderer-backgrounding',
  
  // Stealth - avoid bot detection
  '--disable-blink-features=AutomationControlled',
  '--disable-infobars',
  
  // Window - FORCE visible position
  '--window-size=1280,800',
  '--window-position=50,50',
  '--start-maximized'
];

// Launch browser with all optimizations
async function launchBrowser() {
  utils.log('🚀 Launching Chrome...');
  
  const browserInstance = await puppeteer.launch({
    headless: false,
    userDataDir: config.USER_DATA_DIR,
    defaultViewport: null, // Use window size instead
    args: BROWSER_ARGS,
    ignoreDefaultArgs: ['--enable-automation'], // Hide automation flag
    handleSIGINT: false,  // We handle this ourselves
    handleSIGTERM: false, // We handle this ourselves
    handleSIGHUP: false,  // We handle this ourselves
  });
  
  return browserInstance;
}

// Custom browser title - shows in Chrome window!
const BROWSER_TITLE = '🔥 JMFG UNJANKYOURINSTAGRAM PWND THIS BROWSER 🔥';

// Force set browser window title
async function setBrowserTitle(pageInstance) {
  try {
    await pageInstance.evaluate((title) => {
      document.title = title;
    }, BROWSER_TITLE);
  } catch (e) {}
}

// Create a new page with all optimizations
async function createPage(browserInstance) {
  const newPage = await browserInstance.newPage();
  
  // Stealth: Override navigator.webdriver + Custom title injection
  await newPage.evaluateOnNewDocument((title) => {
    Object.defineProperty(navigator, 'webdriver', { get: () => false });
    // Add fake plugins
    Object.defineProperty(navigator, 'plugins', { get: () => [1, 2, 3, 4, 5] });
    // Add fake languages
    Object.defineProperty(navigator, 'languages', { get: () => ['en-US', 'en'] });
    
    // Override document.title to always show our branding
    let originalTitle = '';
    Object.defineProperty(document, 'title', {
      get: () => title,
      set: (val) => { originalTitle = val; }
    });
    
    // Also set it on load
    window.addEventListener('DOMContentLoaded', () => {
      document.querySelector('title') && (document.querySelector('title').textContent = title);
    });
  }, BROWSER_TITLE);
  
  // Set realistic user agent
  await newPage.setUserAgent('Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36');
  
  // Enable cache
  await newPage.setCacheEnabled(true);
  
  // Block heavy resources
  await newPage.setRequestInterception(true);
  newPage.on('request', (request) => {
    const resourceType = request.resourceType();
    const url = request.url();
    
    // Block heavy resources
    if (['font', 'media'].includes(resourceType)) {
      request.abort();
    }
    // Block analytics/tracking
    else if (url.includes('analytics') || url.includes('tracking') || url.includes('ads')) {
      request.abort();
    }
    else {
      request.continue();
    }
  });
  
  // Handle page crashes
  newPage.on('error', (err) => {
    utils.log(`⚠️ Page error: ${err.message}`);
  });
  
  // Set default timeouts
  newPage.setDefaultTimeout(30000);
  newPage.setDefaultNavigationTimeout(30000);
  
  return newPage;
}

// Check if browser is still healthy
async function isBrowserHealthy(browserInstance) {
  try {
    const pages = await browserInstance.pages();
    return pages.length > 0;
  } catch (e) {
    return false;
  }
}

// Check if page is still responsive
async function isPageHealthy(pageInstance) {
  try {
    await pageInstance.evaluate(() => true);
    return true;
  } catch (e) {
    return false;
  }
}

// Recover from browser crash
async function recoverBrowser() {
  utils.log('🔄 Attempting browser recovery...');
  
  // Close old browser if exists
  try {
    if (browser) await browser.close();
  } catch (e) {}
  
  // Wait a moment
  await utils.sleep(2000);
  
  // Launch fresh browser
  browser = await launchBrowser();
  page = await createPage(browser);
  
  utils.log('✅ Browser recovered!');
  return { browser, page };
}

// Recover from page crash
async function recoverPage() {
  utils.log('🔄 Recovering page...');
  
  try {
    if (page) await page.close();
  } catch (e) {}
  
  page = await createPage(browser);
  
  utils.log('✅ Page recovered!');
  return page;
}

// Clear browser memory periodically - NOW CRASH-PROOF
async function clearMemory(pageInstance) {
  // Use the crash-proof purge system
  // This NEVER throws errors, always completes successfully
  await purgeSystem.safePurge(pageInstance);
}

// Take screenshot for debugging on error
async function captureErrorScreenshot(pageInstance, username) {
  try {
    const screenshotDir = path.join(__dirname, 'error_screenshots');
    if (!fs.existsSync(screenshotDir)) {
      fs.mkdirSync(screenshotDir);
    }
    
    const filename = path.join(screenshotDir, `error_${username}_${Date.now()}.png`);
    await pageInstance.screenshot({ path: filename, fullPage: false });
    utils.log(`📸 Error screenshot saved: ${filename}`);
  } catch (e) {
    // Silent fail
  }
}

// Safe navigation with retry logic (but not too aggressive)
async function safeGoto(pageInstance, url, options = {}) {
  const maxAttempts = 2;
  
  for (let attempt = 1; attempt <= maxAttempts; attempt++) {
    try {
      await pageInstance.goto(url, {
        waitUntil: 'domcontentloaded',
        timeout: 25000,
        ...options
      });
      
      // Set our custom browser title after every navigation!
      await setBrowserTitle(pageInstance);
      
      return true;
    } catch (err) {
      if (attempt === maxAttempts) {
        throw err;
      }
      await utils.sleep(2000);
    }
  }
}

// ============================================
// END CHROME MANAGEMENT SYSTEM
// ============================================

(async () => {
  utils.clearSessionLog();
  utils.log('Starting UnjankyIG...');
  utils.log('Press Ctrl+C to gracefully stop after current user.');

  // ============================================
  // LICENSE CHECK
  // ============================================
  const isPremium = licenseManager.isPremium();
  const remaining = licenseManager.getRemainingUnfollows();
  
  utils.log('');
  utils.log('📜 LICENSE STATUS:');
  if (isPremium) {
    utils.log('   ✅ PREMIUM - Unlimited unfollows');
  } else {
    utils.log(`   🆓 FREE TRIAL - ${remaining}/200 unfollows remaining`);
    
    if (remaining === 0) {
      utils.log('');
      utils.log('╔═══════════════════════════════════════════════════════════╗');
      utils.log('║   🔒 FREE TRIAL EXHAUSTED                                ║');
      utils.log('╚═══════════════════════════════════════════════════════════╝');
      utils.log('');
      utils.log('You have used all 200 free unfollows!');
      utils.log('');
      utils.log('To continue, please upgrade to Premium:');
      utils.log('  ✓ UNLIMITED unfollows');
      utils.log('  ✓ No daily limits');
      utils.log('  ✓ Priority support');
      utils.log('  ✓ Automatic updates');
      utils.log('');
      utils.log('Upgrade now: https://your-payment-page.com');
      utils.log('');
      utils.log('Thank you for trying UnjankyIG! 🙏');
      utils.log('');
      return;
    }
    
    if (remaining < 20) {
      utils.log('   ⚠️ WARNING: Trial almost exhausted!');
      utils.log('   Consider upgrading to Premium soon.');
    }
  }
  utils.log('');

  // Load users from queue
  let users = utils.loadUsers();
  
  if (users.length === 0) {
    utils.log('No users left to process! 🎉');
    return;
  }

  utils.log(`Loaded ${users.length} users.`);

  // Launch browser with Chrome Management System
  browser = await launchBrowser();
  page = await createPage(browser);
  
  utils.log('Navigating to Instagram...');
  await safeGoto(page, 'https://www.instagram.com/');

  // Wait for login or check if already logged in
  try {
    await page.waitForSelector('svg[aria-label="Home"]', { timeout: 5000 });
    utils.log('Already logged in!');
  } catch (e) {
    utils.log('Please log in to Instagram...');
    while (true) {
      try {
        await page.waitForSelector('svg[aria-label="Home"]', { timeout: 5000 });
        utils.log('Login detected!');
        break;
      } catch (e) {
        await utils.sleep(5000);
      }
    }
  }

  // Process users
  let count = 0;
  let skipped = 0;
  let errors = 0;
  const limit = Math.min(users.length, config.DAILY_LIMIT);
  const startingTotal = users.length;
  
  utils.log(`Processing ${limit} users today.`);
  const estMinutes = Math.round((limit * 30) / 60);
  const finishTime = new Date(Date.now() + estMinutes * 60 * 1000);
  utils.log(`Estimated time: ~${estMinutes} minutes (finishing around ${finishTime.toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'})})`);
  utils.log('------------------------------------------------');

  const startTime = Date.now();

  let rateLimitHits = 0;
  const MAX_RATE_LIMITS = 3;
  let consecutiveErrors = 0;
  const MAX_CONSECUTIVE_ERRORS = 10; // More forgiving
  let usersProcessedSinceCleanup = 0;

  while (count < limit && users.length > 0 && !shouldStop) {
    const username = users[0];
    const processed = count + skipped + errors;
    const progress = Math.round((processed / limit) * 100);
    const remaining = limit - processed;
    const avgTimePerUser = processed > 0 ? (Date.now() - startTime) / processed : 30000;
    const etaMs = remaining * avgTimePerUser;
    const etaMinutes = Math.round(etaMs / 1000 / 60);
    const finishTimeStr = new Date(Date.now() + etaMs).toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
    
    process.stdout.write(`[${progress}%] [Done ~${finishTimeStr}] ${username} ... `);

    // Periodic memory cleanup (every 50 users, not too often)
    usersProcessedSinceCleanup++;
    if (usersProcessedSinceCleanup >= 50) {
      try { await clearMemory(page); } catch(e) {}
      usersProcessedSinceCleanup = 0;
    }

    // Health check - but don't go crazy with recovery
    // Only check every so often, not every single user
    if (processed > 0 && processed % 10 === 0) {
      if (!(await isBrowserHealthy(browser))) {
        utils.log('');
        utils.log('🔧 Browser needs restart...');
        try {
          const recovered = await recoverBrowser();
          browser = recovered.browser;
          page = recovered.page;
          await safeGoto(page, 'https://www.instagram.com/');
          await utils.sleep(3000);
        } catch (e) {
          utils.log('❌ Recovery failed, stopping.');
          shouldStop = true;
        }
        continue;
      }
    }

    let actionTaken = false;
    let retries = 0;
    const maxRetries = 3;

    while (retries < maxRetries && !shouldStop) {
      try {
        // Safe navigation with retry logic
        await safeGoto(page, `https://www.instagram.com/${username}/`);
        
        // Wait for the page content to be interactive
        await utils.sleep(1500);

        // Check for rate limiting or blocks
        const pageStatus = await page.evaluate(() => {
          const bodyText = document.body.innerText.toLowerCase();
          if (bodyText.includes('try again later') || bodyText.includes('action blocked')) {
            return 'rate_limited';
          }
          if (bodyText.includes('page not found') || bodyText.includes("sorry, this page isn't available")) {
            return 'not_found';
          }
          if (bodyText.includes('login') && bodyText.includes('sign up')) {
            return 'logged_out';
          }
          return 'ok';
        });

        if (pageStatus === 'rate_limited') {
          rateLimitHits++;
          console.log(`🚫 RATE LIMITED (${rateLimitHits}/${MAX_RATE_LIMITS})`);
          if (rateLimitHits >= MAX_RATE_LIMITS) {
            utils.log('⛔ Too many rate limits! Stopping to protect your account.');
            utils.log('💡 Try again tomorrow or wait a few hours.');
            shouldStop = true;
          } else {
            utils.log('⏸️ Pausing for 5 minutes to cool down...');
            await utils.sleep(300000); // 5 minute cooldown
          }
          break;
        }

        if (pageStatus === 'logged_out') {
          utils.log('');
          utils.log('🔐 Session expired! Please log in again...');
          await safeGoto(page, 'https://www.instagram.com/');
          // Wait for user to log in
          while (true) {
            try {
              await page.waitForSelector('svg[aria-label="Home"]', { timeout: 10000 });
              utils.log('✅ Login restored!');
              break;
            } catch (e) {
              await utils.sleep(5000);
            }
          }
          continue; // Retry this user
        }

        if (pageStatus === 'not_found') {
          console.log('👻 SKIPPED (Account not found)');
          skipped++;
          consecutiveErrors = 0;
          break;
        }

        // Reset counters on successful page load
        rateLimitHits = 0;
        consecutiveErrors = 0;

        const isFollowing = await page.evaluate(() => {
          return document.evaluate("//button[div/div[text()='Following']]", document, null, XPathResult.FIRST_ORDERED_NODE_TYPE, null).singleNodeValue !== null;
        });
        
        if (isFollowing) {
          // Check if we can still unfollow (license check)
          if (!licenseManager.canUnfollowMore()) {
            console.log('🔒 TRIAL LIMIT REACHED');
            utils.log('');
            utils.log('╔═══════════════════════════════════════════════════════════╗');
            utils.log('║   🔒 FREE TRIAL LIMIT REACHED                            ║');
            utils.log('╚═══════════════════════════════════════════════════════════╝');
            utils.log('You have reached 500 free unfollows!');
            utils.log('Please upgrade to Premium to continue.');
            shouldStop = true;
            break;
          }
          
          await page.evaluate(() => {
            document.evaluate("//button[div/div[text()='Following']]", document, null, XPathResult.FIRST_ORDERED_NODE_TYPE, null).singleNodeValue.click();
          });
          
          await utils.sleep(2000);
          
          let confirmed = false;
          for (let i = 0; i < 3; i++) {
            confirmed = await page.evaluate(() => {
              let btn = document.evaluate("//*[text()='Unfollow']", document, null, XPathResult.FIRST_ORDERED_NODE_TYPE, null).singleNodeValue;
              if (btn) {
                btn.click();
                return true;
              }
              return false;
            });
            
            if (confirmed) break;
            await utils.sleep(1000);
          }
          
          if (confirmed) {
            console.log('✅ UNFOLLOWED');
            actionTaken = true;
            count++;
            
            // Increment the global unfollow counter (for license enforcement)
            const newCount = licenseManager.incrementUnfollowCount();
            const remaining = licenseManager.getRemainingUnfollows();
            if (!licenseManager.isPremium() && remaining <= 10 && remaining > 0) {
              console.log(`   ⚠️ Only ${remaining} free unfollows remaining!`);
            }
          } else {
            console.log('⚠️ CONFIRM MISSING');
            await captureErrorScreenshot(page, username);
            errors++;
          }

        } else {
          console.log('⏭️ SKIPPED (Already done)');
          skipped++;
        }

        break; // Success, exit retry loop

      } catch (err) {
        retries++;
        
        if (retries >= maxRetries) {
          console.log(`❌ ERROR: ${err.message.substring(0, 50)}`);
          errors++;
          consecutiveErrors++;
          
          // Too many consecutive FINAL errors - stop
          if (consecutiveErrors >= MAX_CONSECUTIVE_ERRORS) {
            utils.log('');
            utils.log('⚠️ Too many errors in a row. Taking a break...');
            await utils.sleep(60000); // Wait 1 minute
            consecutiveErrors = 0; // Reset and try again
          }
        } else {
          process.stdout.write(`⚠️ Retry ${retries}/${maxRetries}... `);
          await utils.sleep(3000);
        }
      }
    }

    // Always remove from queue and add to processed
    utils.markProcessed(username);
    users.shift();
    utils.saveUsers(users);

    // Delay between actions
    if (count < limit && users.length > 0 && !shouldStop) {
      if (actionTaken) {
        const delay = Math.floor(Math.random() * (config.MAX_DELAY - config.MIN_DELAY + 1)) + config.MIN_DELAY;
        const delaySecs = Math.round(delay / 1000);
        process.stdout.write(`    ⏳ Waiting ${delaySecs}s...`);
        await utils.sleep(delay);
        process.stdout.clearLine && process.stdout.clearLine(0);
        process.stdout.cursorTo && process.stdout.cursorTo(0);
      } else {
        await utils.sleep(2000);
      }
    }
  }

  // Handle graceful stop
  if (shouldStop) {
    utils.log('');
    utils.log('🛑 Stopped by user request.');
  }

  const elapsed = Math.round((Date.now() - startTime) / 1000 / 60);
  const totalProcessed = count + skipped + errors;
  const overallProgress = Math.round((startingTotal - users.length) / startingTotal * 100);
  
  utils.log('------------------------------------------------');
  utils.log('🎉 DAILY RUN COMPLETE!');
  utils.log(`✅ Unfollowed: ${count}`);
  utils.log(`⏭️ Skipped:    ${skipped}`);
  utils.log(`❌ Errors:     ${errors}`);
  utils.log(`⏱️ Time:       ${elapsed} minutes`);
  utils.log(`📊 Remaining:  ${users.length} users`);
  utils.log('------------------------------------------------');
  
  // Generate report
  const reportFile = generateDailyReport({
    unfollowed: count,
    skipped: skipped,
    errors: errors,
    timeMinutes: elapsed,
    progressPercent: overallProgress
  });
  
  utils.log(`📄 Report saved: ${reportFile}`);
  utils.log('------------------------------------------------');
  
  // Show purge system stats
  purgeSystem.printStats();
  
  utils.log('Daelytes Unjank Your IG Program Made With Love & Antigravity');
  utils.log('------------------------------------------------');
  
  // Play completion sound (Windows tada!)
  try {
    exec('powershell -c "(New-Object Media.SoundPlayer \'C:\\Windows\\Media\\tada.wav\').PlaySync()"');
    utils.log('🔔 Ding! All done!');
  } catch (e) {
    // Sound failed, try console beep as fallback
    process.stdout.write('\x07');
  }
  
  // Auto-open the report in browser
  try {
    exec(`start "" "${reportFile}"`);
    utils.log('📊 Opening report in browser...');
  } catch (e) {
    // Failed to open, no big deal
  }
  
  utils.log('Closing browser in 5 seconds...');
  await utils.sleep(5000);
  await browser.close();
})();
