/**
 * UnjankyIG - CSV Import Handler
 * Imports a CSV file with Instagram usernames
 */

const fs = require('fs');
const path = require('path');
const csv = require('csv-parser');
const config = require('./config');

const inputFile = process.argv[2] || 'imported_list.csv';

if (!fs.existsSync(inputFile)) {
    console.error(`❌ Error: File not found: ${inputFile}`);
    process.exit(1);
}

console.log(`\n📁 Importing from: ${inputFile}\n`);

// Load already processed users to filter them out
const loadProcessed = () => {
  if (!fs.existsSync(config.PROCESSED_FILE)) {
    return new Set();
  }
  const content = fs.readFileSync(config.PROCESSED_FILE, 'utf-8');
  return new Set(
    content.split('\n')
      .map(line => line.split('\t')[0].trim().toLowerCase())
      .filter(u => u.length > 0)
  );
};

// Load existing queue to preserve order and avoid duplicates
const loadExistingQueue = () => {
  if (!fs.existsSync(config.USERS_FILE)) {
    return new Set();
  }
  const content = fs.readFileSync(config.USERS_FILE, 'utf-8');
  return new Set(
    content.split('\n')
      .map(u => u.trim().toLowerCase())
      .filter(u => u.length > 0)
  );
};

const processedUsers = loadProcessed();
const existingQueue = loadExistingQueue();
const usernames = [];
const seen = new Set();

let totalRead = 0;
let duplicates = 0;
let alreadyProcessed = 0;
let alreadyInQueue = 0;

fs.createReadStream(inputFile)
  .pipe(csv({ headers: false }))
  .on('data', (row) => {
    const rawUsername = row['0']; 
    if (rawUsername && rawUsername.trim() !== '') {
      totalRead++;
      const username = rawUsername.trim();
      const usernameLower = username.toLowerCase();
      
      if (seen.has(usernameLower)) {
        duplicates++;
        return;
      }
      seen.add(usernameLower);
      
      if (processedUsers.has(usernameLower)) {
        alreadyProcessed++;
        return;
      }
      
      if (existingQueue.has(usernameLower)) {
        alreadyInQueue++;
        return;
      }
      
      usernames.push(username);
    }
  })
  .on('end', () => {
    const existingUsers = fs.existsSync(config.USERS_FILE) 
      ? fs.readFileSync(config.USERS_FILE, 'utf-8').split('\n').filter(u => u.trim().length > 0)
      : [];
    
    const finalList = [...existingUsers, ...usernames];
    fs.writeFileSync(config.USERS_FILE, finalList.join('\n'));
    
    console.log('━'.repeat(50));
    console.log('\n📊 Import Summary:\n');
    console.log(`   Total in file:      ${totalRead}`);
    console.log(`   Duplicates:         ${duplicates} (skipped)`);
    console.log(`   Already processed:  ${alreadyProcessed} (skipped)`);
    console.log(`   Already in queue:   ${alreadyInQueue} (skipped)`);
    console.log(`   New users added:    ${usernames.length}`);
    console.log(`   Total in queue:     ${finalList.length}`);
    console.log('\n✅ Import complete!\n');
  })
  .on('error', (error) => {
    console.error('❌ Error reading CSV:', error.message);
    process.exit(1);
  });

