const express = require('express');
const path = require('path');
const { exec } = require('child_process');
const fs = require('fs');
const licenseManager = require('./license_manager');
require('dotenv').config();

// Stripe Payment Service
let stripePayment;
try {
  stripePayment = require('./stripe_payment_service');
  console.log('✅ Stripe payment service loaded');
} catch (error) {
  console.warn('⚠️  Stripe payment service not available:', error.message);
}

const app = express();
const PORT = process.env.PORT || 3456;

// Middleware
app.use(express.json());
app.use(express.urlencoded({ extended: true }));
app.use(express.static('public'));

// State
let botRunning = false;
let botProcess = null;

// ============================================
// API ENDPOINTS
// ============================================

// Get license status
app.get('/api/status', (req, res) => {
  const isPremium = licenseManager.isPremium();
  const remaining = licenseManager.getRemainingUnfollows();
  const hasUsers = fs.existsSync('users.txt');
  const userCount = hasUsers ? fs.readFileSync('users.txt', 'utf8').split('\n').filter(l => l.trim()).length : 0;
  
  res.json({
    isPremium,
    remaining,
    totalLimit: 200,
    hasUsers,
    userCount,
    botRunning
  });
});

// Auto-fetch non-followers (Instagram scraping)
app.post('/api/fetch-list', (req, res) => {
  if (botRunning) {
    return res.json({ success: false, error: 'Bot is already running' });
  }
  
  console.log('🤖 Starting auto-fetch from Instagram...');
  
  const fetchProcess = exec('node auto_fetch_nonfollowers.js', (error, stdout, stderr) => {
    if (error) {
      console.error('❌ Fetch error:', error);
      return;
    }
    console.log('✅ Auto-fetch complete!');
  });
  
  res.json({ 
    success: true, 
    message: 'Fetching non-followers from Instagram... Browser will open. Please log in if needed. This takes 2-5 minutes.' 
  });
});

// Import CSV list (user provided)
app.post('/api/import-list', (req, res) => {
  const { csvContent } = req.body;
  
  if (!csvContent) {
    return res.json({ success: false, error: 'No CSV content provided' });
  }
  
  try {
    console.log('📁 Importing CSV list...');
    
    // Save CSV
    fs.writeFileSync('imported_list.csv', csvContent);
    
    // Process it
    exec('node import_csv.js imported_list.csv', (error, stdout, stderr) => {
      if (error) {
        console.error('❌ Import error:', error);
        return res.json({ success: false, error: 'Failed to process CSV' });
      }
      console.log(stdout);
      console.log('✅ CSV import complete!');
    });
    
    res.json({ success: true, message: 'List imported successfully!' });
  } catch (e) {
    res.json({ success: false, error: e.message });
  }
});

// Start bot
app.post('/api/start', (req, res) => {
  if (botRunning) {
    return res.json({ success: false, error: 'Bot is already running' });
  }
  
  if (!fs.existsSync('users.txt')) {
    return res.json({ success: false, error: 'No users list found. Please fetch or import a list first.' });
  }
  
  console.log('Starting bot...');
  botRunning = true;
  
  botProcess = exec('node index.js', (error, stdout, stderr) => {
    botRunning = false;
    botProcess = null;
    console.log('Bot finished');
  });
  
  res.json({ success: true, message: 'Bot started!' });
});

// Stop bot
app.post('/api/stop', (req, res) => {
  if (!botRunning || !botProcess) {
    return res.json({ success: false, error: 'Bot is not running' });
  }
  
  botProcess.kill();
  botRunning = false;
  botProcess = null;
  
  res.json({ success: true, message: 'Bot stopped' });
});

// Create Stripe payment session
app.post('/api/create-payment', async (req, res) => {
  try {
    if (!stripePayment) {
      return res.status(500).json({ 
        success: false, 
        error: 'Stripe payment service not configured' 
      });
    }

    const { platform, email } = req.body;
    const session = await stripePayment.createPaymentSession(
      platform || 'WINDOWS',
      email || null
    );

    res.json({
      success: true,
      sessionId: session.sessionId,
      url: session.url,
      licenseKey: session.licenseKey,
    });
  } catch (error) {
    console.error('❌ Payment creation error:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Verify payment session
app.get('/api/verify-payment/:sessionId', async (req, res) => {
  try {
    if (!stripePayment) {
      return res.status(500).json({ 
        success: false, 
        error: 'Stripe payment service not configured' 
      });
    }

    const result = await stripePayment.verifyPaymentSession(req.params.sessionId);
    res.json(result);
  } catch (error) {
    console.error('❌ Payment verification error:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// Activate license
app.post('/api/activate', (req, res) => {
  const { licenseKey } = req.body;
  
  if (!licenseKey) {
    return res.json({ success: false, error: 'License key required' });
  }
  
  try {
    const activate = require('./activate_license');
    const result = activate.activateLicense(licenseKey);
    res.json(result);
  } catch (error) {
    res.status(500).json({ success: false, error: error.message });
  }
});

// Payment success page
app.get('/payment-success', (req, res) => {
  const { session_id, key } = req.query;
  res.send(`
    <!DOCTYPE html>
    <html>
    <head>
      <title>Payment Successful - UnjankyIG</title>
      <style>
        body {
          font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
          background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
          display: flex;
          align-items: center;
          justify-content: center;
          height: 100vh;
          margin: 0;
        }
        .success-box {
          background: white;
          padding: 40px;
          border-radius: 20px;
          box-shadow: 0 20px 60px rgba(0,0,0,0.3);
          text-align: center;
          max-width: 500px;
        }
        .checkmark {
          font-size: 80px;
          color: #4CAF50;
          margin-bottom: 20px;
        }
        h1 { color: #333; margin-bottom: 10px; }
        .license-key {
          background: #f5f5f5;
          padding: 15px;
          border-radius: 10px;
          font-family: monospace;
          font-size: 18px;
          font-weight: bold;
          color: #667eea;
          margin: 20px 0;
          word-break: break-all;
        }
        .btn {
          background: #667eea;
          color: white;
          padding: 12px 30px;
          border: none;
          border-radius: 10px;
          font-size: 16px;
          cursor: pointer;
          text-decoration: none;
          display: inline-block;
          margin-top: 20px;
        }
        .btn:hover { background: #5568d3; }
      </style>
    </head>
    <body>
      <div class="success-box">
        <div class="checkmark">✓</div>
        <h1>Payment Successful!</h1>
        <p>Thank you for purchasing UnjankyIG Premium!</p>
        <div class="license-key">${key || 'Loading...'}</div>
        <p style="color: #666; font-size: 14px;">Save this license key! Auto-activating now...</p>
        <a href="/" class="btn">Return to Dashboard</a>
      </div>
      <script>
        if ('${key}') {
          fetch('/api/activate', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ licenseKey: '${key}' })
          }).then(r => r.json()).then(data => {
            if (data.success) console.log('✅ License auto-activated!');
          });
        }
      </script>
    </body>
    </html>
  `);
});

// Payment cancelled page
app.get('/payment-cancelled', (req, res) => {
  res.send(`
    <!DOCTYPE html>
    <html>
    <head>
      <title>Payment Cancelled</title>
      <style>
        body {
          font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
          background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
          display: flex;
          align-items: center;
          justify-content: center;
          height: 100vh;
          margin: 0;
        }
        .cancel-box {
          background: white;
          padding: 40px;
          border-radius: 20px;
          box-shadow: 0 20px 60px rgba(0,0,0,0.3);
          text-align: center;
          max-width: 500px;
        }
        .icon { font-size: 80px; margin-bottom: 20px; }
        h1 { color: #333; margin-bottom: 10px; }
        .btn {
          background: #667eea;
          color: white;
          padding: 12px 30px;
          border: none;
          border-radius: 10px;
          font-size: 16px;
          cursor: pointer;
          text-decoration: none;
          display: inline-block;
          margin-top: 20px;
        }
        .btn:hover { background: #5568d3; }
      </style>
    </head>
    <body>
      <div class="cancel-box">
        <div class="icon">😕</div>
        <h1>Payment Cancelled</h1>
        <p>No worries! You still have 200 free unfollows to try.</p>
        <p style="color: #666; font-size: 14px;">Come back when you're ready to upgrade!</p>
        <a href="/" class="btn">Return to Dashboard</a>
      </div>
    </body>
    </html>
  `);
});

// ============================================
// SERVE HTML
// ============================================

app.get('/', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'dashboard.html'));
});

// ============================================
// START SERVER
// ============================================

app.listen(PORT, () => {
  console.log(`\n╔═══════════════════════════════════════════════════════════╗`);
  console.log(`║                                                           ║`);
  console.log(`║         🔥 UNJANKY IG DASHBOARD 🔥                       ║`);
  console.log(`║                                                           ║`);
  console.log(`╚═══════════════════════════════════════════════════════════╝\n`);
  console.log(`   Dashboard running at: http://localhost:${PORT}`);
  console.log(`   Opening in your browser...\n`);
  
  // Auto-open browser
  const url = `http://localhost:${PORT}`;
  const start = process.platform === 'darwin' ? 'open' : process.platform === 'win32' ? 'start' : 'xdg-open';
  exec(`${start} ${url}`);
});

