/**
 * UnjankyIG - Auto-Fetch Non-Followers
 * Automatically logs into Instagram and finds who doesn't follow you back
 */

const puppeteer = require('puppeteer');
const fs = require('fs');
const path = require('path');
const config = require('./config');

console.log('\n🤖 UnjankyIG - Auto-Fetch Non-Followers\n');

// Delay helper
const delay = (ms) => new Promise(resolve => setTimeout(resolve, ms));

// Random delay between min and max
const randomDelay = (min, max) => delay(Math.floor(Math.random() * (max - min + 1)) + min);

async function scrollToLoadAll(page, selector, maxScrolls = 50) {
  console.log('📜 Scrolling to load all users...');
  
  let previousHeight = 0;
  let scrollCount = 0;
  let noChangeCount = 0;
  
  while (scrollCount < maxScrolls && noChangeCount < 3) {
    const currentHeight = await page.evaluate((sel) => {
      const container = document.querySelector(sel);
      if (container) {
        container.scrollTop = container.scrollHeight;
        return container.scrollHeight;
      }
      return 0;
    }, selector);
    
    if (currentHeight === previousHeight) {
      noChangeCount++;
    } else {
      noChangeCount = 0;
      previousHeight = currentHeight;
    }
    
    scrollCount++;
    await randomDelay(1000, 2000);
    
    if (scrollCount % 10 === 0) {
      console.log(`   Scrolled ${scrollCount} times...`);
    }
  }
  
  console.log(`✅ Finished scrolling (${scrollCount} scrolls)\n`);
}

async function extractUsernames(page, listType) {
  console.log(`📋 Extracting ${listType} list...`);
  
  await delay(2000);
  
  const usernames = await page.evaluate(() => {
    const users = new Set();
    
    // Try multiple selectors for username links
    const selectors = [
      'a[href^="/"]',
      'a[role="link"]',
      'span._ap3a._aaco._aacw._aacx._aad7._aade'
    ];
    
    selectors.forEach(selector => {
      const links = document.querySelectorAll(selector);
      links.forEach(link => {
        const href = link.getAttribute('href');
        if (href && href.startsWith('/') && href !== '/' && !href.includes('/p/') && !href.includes('/reel/')) {
          const username = href.replace(/\//g, '').trim();
          if (username && username.length > 0 && username.length < 31) {
            users.add(username);
          }
        }
        
        // Also try text content
        const text = link.textContent?.trim();
        if (text && text.length > 0 && text.length < 31 && !text.includes(' ')) {
          users.add(text);
        }
      });
    });
    
    return Array.from(users);
  });
  
  console.log(`✅ Found ${usernames.length} ${listType}\n`);
  return usernames;
}

async function getFollowers(page, username) {
  console.log('👥 Getting followers list...\n');
  
  const followersUrl = `https://www.instagram.com/${username}/followers/`;
  await page.goto(followersUrl, { waitUntil: 'networkidle2', timeout: 60000 });
  await delay(3000);
  
  // Wait for followers dialog
  const dialogSelector = 'div[role="dialog"]';
  await page.waitForSelector(dialogSelector, { timeout: 10000 });
  
  // Scroll to load all
  await scrollToLoadAll(page, `${dialogSelector} > div > div:last-child`);
  
  return extractUsernames(page, 'followers');
}

async function getFollowing(page, username) {
  console.log('👤 Getting following list...\n');
  
  const followingUrl = `https://www.instagram.com/${username}/following/`;
  await page.goto(followingUrl, { waitUntil: 'networkidle2', timeout: 60000 });
  await delay(3000);
  
  // Wait for following dialog
  const dialogSelector = 'div[role="dialog"]';
  await page.waitForSelector(dialogSelector, { timeout: 10000 });
  
  // Scroll to load all
  await scrollToLoadAll(page, `${dialogSelector} > div > div:last-child`);
  
  return extractUsernames(page, 'following');
}

async function main() {
  let browser;
  
  try {
    console.log('🚀 Launching browser...\n');
    
    browser = await puppeteer.launch({
      headless: false, // Show browser so you can log in
      userDataDir: config.USER_DATA_DIR,
      args: [
        '--no-sandbox',
        '--disable-setuid-sandbox',
        '--disable-dev-shm-usage',
        '--disable-blink-features=AutomationControlled'
      ]
    });
    
    const page = await browser.newPage();
    await page.setViewport({ width: 1280, height: 800 });
    
    // Go to Instagram
    console.log('📱 Opening Instagram...\n');
    await page.goto('https://www.instagram.com/', { waitUntil: 'networkidle2' });
    await delay(3000);
    
    // Check if already logged in
    const isLoggedIn = await page.evaluate(() => {
      return !!document.querySelector('a[href*="/direct/"]') || 
             !!document.querySelector('svg[aria-label="Home"]');
    });
    
    if (!isLoggedIn) {
      console.log('🔐 Please log in to Instagram...');
      console.log('⏳ Waiting for you to log in (up to 2 minutes)...\n');
      
      // Wait for login (look for home/DM icons)
      await page.waitForSelector('a[href*="/direct/"], svg[aria-label="Home"]', { 
        timeout: 120000 
      });
      
      console.log('✅ Logged in!\n');
      await delay(3000);
    } else {
      console.log('✅ Already logged in!\n');
    }
    
    // Get current username
    const username = await page.evaluate(() => {
      const profileLink = document.querySelector('a[href*="/"] img[alt*="profile picture"]');
      if (profileLink) {
        const href = profileLink.closest('a')?.getAttribute('href');
        return href?.replace(/\//g, '');
      }
      return null;
    });
    
    if (!username) {
      throw new Error('Could not detect your username. Please make sure you are logged in.');
    }
    
    console.log(`📍 Your username: @${username}\n`);
    console.log('━'.repeat(50));
    console.log('\n');
    
    // Get followers
    const followers = await getFollowers(page, username);
    
    // Get following
    const following = await getFollowing(page, username);
    
    // Calculate non-followers
    console.log('🔍 Calculating non-followers...\n');
    const followersSet = new Set(followers.map(u => u.toLowerCase()));
    const nonFollowers = following.filter(u => !followersSet.has(u.toLowerCase()));
    
    console.log('━'.repeat(50));
    console.log('\n📊 RESULTS:\n');
    console.log(`   Followers:      ${followers.length}`);
    console.log(`   Following:      ${following.length}`);
    console.log(`   Non-followers:  ${nonFollowers.length}`);
    console.log('');
    
    // Save to file
    if (nonFollowers.length > 0) {
      const outputFile = config.USERS_FILE || 'users.txt';
      fs.writeFileSync(outputFile, nonFollowers.join('\n'));
      console.log(`✅ Saved to: ${outputFile}\n`);
    } else {
      console.log('🎉 Everyone follows you back!\n');
    }
    
  } catch (error) {
    console.error('\n❌ Error:', error.message);
    console.error('\nIf you got a timeout, the script might need more time.');
    console.error('Try running it again!\n');
  } finally {
    if (browser) {
      await browser.close();
    }
  }
}

// Run
main().catch(console.error);

